/*
 * Decompiled with CFR 0.152.
 */
package com.gmail.nossr50.database;

import com.gmail.nossr50.database.ExpectedType;
import com.gmail.nossr50.database.FlatFileDataFlag;
import com.gmail.nossr50.database.flatfile.FlatFileDataBuilder;
import com.gmail.nossr50.database.flatfile.FlatFileDataContainer;
import com.gmail.nossr50.database.flatfile.FlatFileDataUtil;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.UUID;
import java.util.logging.Logger;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class FlatFileDataProcessor {
    @NotNull
    private final List<FlatFileDataContainer> flatFileDataContainers;
    @NotNull
    private final List<FlatFileDataFlag> flatFileDataFlags;
    @NotNull
    private final Logger logger;
    private final HashSet<String> names;
    private final HashSet<UUID> uuids;
    private int uniqueProcessingID;
    boolean corruptDataFound;

    public FlatFileDataProcessor(@NotNull Logger logger) {
        this.logger = logger;
        this.flatFileDataContainers = new ArrayList<FlatFileDataContainer>();
        this.flatFileDataFlags = new ArrayList<FlatFileDataFlag>();
        this.names = new HashSet();
        this.uuids = new HashSet();
        this.uniqueProcessingID = 0;
    }

    public void processData(@NotNull String lineData) {
        assert (!lineData.isEmpty());
        if (lineData.charAt(lineData.length() - 1) != ':') {
            lineData = lineData.concat(":");
        }
        String[] splitDataLine = lineData.split(":");
        FlatFileDataBuilder builder = new FlatFileDataBuilder(splitDataLine, this.uniqueProcessingID);
        ++this.uniqueProcessingID;
        boolean[] badDataValues = new boolean[55];
        boolean anyBadData = false;
        if (splitDataLine.length < this.getMinimumSplitDataLength()) {
            if (!this.corruptDataFound) {
                this.logger.severe("Some corrupt data was found in mcmmo.users and has been repaired, it is possible that some player data has been lost in this process.");
                this.corruptDataFound = true;
            }
            builder.appendFlag(FlatFileDataFlag.CORRUPTED_OR_UNRECOGNIZABLE);
            if (splitDataLine.length >= 10 && splitDataLine[0] != null && !splitDataLine[0].isEmpty() && splitDataLine[0].length() <= 16 && splitDataLine[0].length() >= 3) {
                this.logger.severe("Not enough data found to recover corrupted player data for user: " + splitDataLine[0]);
                this.registerData(builder.appendFlag(FlatFileDataFlag.TOO_INCOMPLETE));
                return;
            }
            this.registerData(builder.appendFlag(FlatFileDataFlag.CORRUPTED_OR_UNRECOGNIZABLE));
            return;
        }
        boolean invalidUUID = false;
        String name = splitDataLine[0];
        String strOfUUID = splitDataLine[41];
        if (name.isEmpty()) {
            this.reportBadDataLine("No name found for data", "[MISSING NAME]", lineData);
            builder.appendFlag(FlatFileDataFlag.MISSING_NAME);
            anyBadData = true;
            badDataValues[0] = true;
        }
        if (strOfUUID.isEmpty() || strOfUUID.equalsIgnoreCase("NULL")) {
            invalidUUID = true;
            badDataValues[41] = true;
            this.reportBadDataLine("Empty/null UUID for user", "Empty/null", lineData);
            builder.appendFlag(FlatFileDataFlag.BAD_UUID_DATA);
            anyBadData = true;
        }
        UUID uuid = null;
        try {
            uuid = UUID.fromString(strOfUUID);
        }
        catch (IllegalArgumentException e) {
            invalidUUID = true;
            badDataValues[41] = true;
            this.reportBadDataLine("Invalid UUID data found for user", strOfUUID, lineData);
            builder.appendFlag(FlatFileDataFlag.BAD_UUID_DATA);
        }
        if (!invalidUUID && uuid != null && this.uuids.contains(uuid)) {
            this.registerData(builder.appendFlag(FlatFileDataFlag.DUPLICATE_UUID));
            return;
        }
        this.uuids.add(uuid);
        if (this.names.contains(name)) {
            builder.appendFlag(FlatFileDataFlag.DUPLICATE_NAME);
            anyBadData = true;
            badDataValues[0] = true;
        }
        if (!name.isEmpty()) {
            this.names.add(name);
        }
        splitDataLine = this.isDataSchemaUpToDate(splitDataLine, builder, badDataValues);
        for (int i = 0; i < 55; ++i) {
            if (this.shouldNotBeEmpty(splitDataLine[i], i)) {
                if (i == 44) {
                    builder.appendFlag(FlatFileDataFlag.LAST_LOGIN_SCHEMA_UPGRADE);
                }
                badDataValues[i] = true;
                anyBadData = true;
                continue;
            }
            boolean isCorrectType = this.isOfExpectedType(splitDataLine[i], FlatFileDataProcessor.getExpectedValueType(i));
            if (isCorrectType) continue;
            anyBadData = true;
            badDataValues[i] = true;
        }
        if (anyBadData) {
            builder.appendFlag(FlatFileDataFlag.BAD_VALUES);
            builder.appendBadDataValues(badDataValues);
        }
        this.registerData(builder);
    }

    @NotNull
    public String[] isDataSchemaUpToDate(@NotNull String[] splitDataLine, @NotNull FlatFileDataBuilder builder, boolean[] badDataValues) {
        assert (splitDataLine.length <= 55);
        if (splitDataLine.length < 55) {
            int oldLength = splitDataLine.length;
            splitDataLine = Arrays.copyOf(splitDataLine, 55);
            int newLength = splitDataLine.length;
            for (int i = oldLength; i < newLength - 1; ++i) {
                badDataValues[i] = true;
            }
            builder.appendFlag(FlatFileDataFlag.INCOMPLETE);
            builder.setSplitStringData(splitDataLine);
        }
        return splitDataLine;
    }

    public boolean shouldNotBeEmpty(@Nullable String data, int index) {
        if (FlatFileDataProcessor.getExpectedValueType(index) == ExpectedType.IGNORED) {
            return false;
        }
        return data == null || data.isEmpty();
    }

    public boolean isOfExpectedType(@NotNull String data, @NotNull ExpectedType expectedType) {
        switch (expectedType) {
            case STRING: {
                return true;
            }
            case INTEGER: {
                try {
                    Integer.valueOf(data);
                    return true;
                }
                catch (Exception e) {
                    return false;
                }
            }
            case BOOLEAN: {
                return data.equalsIgnoreCase("true") || data.equalsIgnoreCase("false");
            }
            case FLOAT: {
                try {
                    Float.valueOf(data);
                    return true;
                }
                catch (NumberFormatException e) {
                    return false;
                }
            }
            case DOUBLE: {
                try {
                    Double.valueOf(data);
                    return true;
                }
                catch (NumberFormatException e) {
                    return false;
                }
            }
            case UUID: {
                try {
                    UUID.fromString(data);
                    return true;
                }
                catch (IllegalArgumentException e) {
                    return false;
                }
            }
            case OUT_OF_RANGE: {
                throw new ArrayIndexOutOfBoundsException("Value matched type OUT_OF_RANGE, this should never happen.");
            }
        }
        return true;
    }

    private void reportBadDataLine(String warning, String context, String dataLine) {
        this.logger.warning("FlatFileDatabaseBuilder Warning: " + warning + " - " + context);
        this.logger.warning("FlatFileDatabaseBuilder: (Line Data) - " + dataLine);
        this.logger.warning("mcMMO will repair this data if automatically (if it is possible).");
    }

    private int getMinimumSplitDataLength() {
        return 42;
    }

    private void registerData(@NotNull FlatFileDataBuilder builder) {
        FlatFileDataContainer flatFileDataContainer = builder.build();
        this.flatFileDataContainers.add(flatFileDataContainer);
        if (flatFileDataContainer.getDataFlags() != null) {
            this.flatFileDataFlags.addAll(flatFileDataContainer.getDataFlags());
        }
    }

    @NotNull
    public static ExpectedType getExpectedValueType(int dataIndex) throws IndexOutOfBoundsException {
        return switch (dataIndex) {
            case 0 -> ExpectedType.STRING;
            case 2, 3, 23, 33, 37, 38 -> ExpectedType.IGNORED;
            case 1, 5, 7, 8, 9, 10, 11, 12, 13, 14, 24, 26, 27, 28, 29, 30, 31, 32, 34, 36, 39, 42, 43, 46, 48, 49, 50, 51, 53, 54 -> ExpectedType.INTEGER;
            case 4, 6, 15, 16, 17, 18, 19, 20, 21, 22, 25, 35, 40, 45, 47, 52 -> ExpectedType.FLOAT;
            case 41 -> ExpectedType.UUID;
            case 44 -> ExpectedType.LONG;
            default -> throw new IndexOutOfBoundsException();
        };
    }

    @NotNull
    public List<FlatFileDataContainer> getFlatFileDataContainers() {
        return this.flatFileDataContainers;
    }

    @NotNull
    public List<FlatFileDataFlag> getFlatFileDataFlags() {
        return this.flatFileDataFlags;
    }

    public int getDataFlagCount() {
        return this.flatFileDataFlags.size();
    }

    @NotNull
    public StringBuilder processDataForSave() {
        StringBuilder stringBuilder = new StringBuilder();
        for (FlatFileDataContainer dataContainer : this.flatFileDataContainers) {
            Object[] splitData = FlatFileDataUtil.getPreparedSaveDataLine(dataContainer);
            if (splitData == null) continue;
            String fromSplit = StringUtils.join((Object[])splitData, (String)":") + ":";
            stringBuilder.append(fromSplit).append("\r\n");
        }
        return stringBuilder;
    }
}

